<?php

namespace App\Controller;

use App\Entity\Pokemon;
use App\Repository\PokemonRepository;
use App\Model\SearchData;
use App\Form\PokemonType;
use App\Form\SearchPokemonType;
use App\Model\PokemonSearchAvancee;
use App\Form\SearchPokemonAvanceeType;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;

/**
 * @Route("/pokemon")
 */
class PokemonController extends AbstractController
{
    /**
     * @Route("/recherche", name="pokemon_recherche", methods={"GET"})
     * @IsGranted("ROLE_USER")
     */
    public function rechercheAvancee(PokemonRepository $pokemonRepository, Request $request): Response
    {
        $search = new PokemonSearchAvancee();
        $form = $this->createForm(SearchPokemonAvanceeType::class,$search);
        $form->handleRequest($request);
        
        if($search->getMinVie()>$search->getMaxVie() || $search->getMinAttaque()>$search->getMaxAttaque() || $search->getMinDefense()>$search->getMaxDefense()){
            echo ("<script LANGUAGE='JavaScript'>
                window.alert('Veuillez vérifier vos valeurs, une valeur minimale ne peut être supérieure à une valeur maximale');
                window.location.href='/pokemon/recherche';
                </script>");
        }

        $pokemons = $pokemonRepository->findSearchFilter($search);
        return $this->render('pokemon/_avancee.html.twig', [
            'pokemons' => $pokemons,
            'form' => $form->createView()
        ]);
    }
    
    /**
     * @Route("/", name="pokemon_index", methods={"GET"})
     * @IsGranted("ROLE_USER")
     */
    public function index(PokemonRepository $pokemonRepository, Request $request): Response
    {
        $data = new SearchData();
        $form = $this->createForm(SearchPokemonType::class, $data);
        $form->handleRequest($request);

        if(!empty($data->q)){
            $pokemon = $pokemonRepository->findSearch($data);
        }else{
            $pokemon = $pokemonRepository->findAll();
        }
                
        return $this->render('pokemon/index.html.twig', [
            'pokemon' => $pokemon,
            'form' => $form->createView()
        ]);
    }
    
    /**
     * @Route("/new", name="pokemon_new", methods={"GET","POST"})
     * @IsGranted("ROLE_ADMIN")
     */
    public function new(Request $request): Response
    {
        $pokemon = new Pokemon();
        $form = $this->createForm(PokemonType::class, $pokemon);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $entityManager = $this->getDoctrine()->getManager();
            $entityManager->persist($pokemon);
            $entityManager->flush();

            return $this->redirectToRoute('pokemon_index');
        }

        return $this->render('pokemon/new.html.twig', [
            'pokemon' => $pokemon,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @Route("/{id}", name="pokemon_show", methods={"GET"})
     * @IsGranted("ROLE_USER")
     */
    public function show(Pokemon $pokemon): Response
    {
        return $this->render('pokemon/show.html.twig', [
            'pokemon' => $pokemon,
        ]);
    }

    /**
     * @Route("/{id}/edit", name="pokemon_edit", methods={"GET","POST"})
     * @IsGranted("ROLE_ADMIN")
     */
    public function edit(Request $request, Pokemon $pokemon): Response
    {
        $form = $this->createForm(PokemonType::class, $pokemon);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->getDoctrine()->getManager()->flush();

            return $this->redirectToRoute('pokemon_index');
        }

        return $this->render('pokemon/edit.html.twig', [
            'pokemon' => $pokemon,
            'form' => $form->createView(),
        ]);
    }

    /**
     * @Route("/{id}", name="pokemon_delete", methods={"POST"})
     * @IsGranted("ROLE_ADMIN")
     */
    public function delete(Request $request, Pokemon $pokemon): Response
    {
        if ($this->isCsrfTokenValid('delete'.$pokemon->getId(), $request->request->get('_token'))) {
            $entityManager = $this->getDoctrine()->getManager();
            $entityManager->remove($pokemon);
            $entityManager->flush();
        }

        return $this->redirectToRoute('pokemon_index');
    }
}
