const db = require("../models/index");
const UserProject = db.userProject;
const Project = db.project;
const UserSkill = db.userSkill;
const Skill = db.skill;

// Update user's skill note
exports.update = async (req, res) => {
    // Check if request body is empty
    if (Object.keys(req.body).length === 0) {
        return res.status(400).send({ message: "No data was provided" });
    }

    const listNotes = ["Non acquis", "En cours d'obtention", "Validé"];
    // Check if the note provided is in the list of acceptable notes
    if (!listNotes.includes(req.body.note)) {
        delete req.body.note; // Remove the note from the request body
        return res.status(400).send({ message: "Your evaluation note is not accepted" });
    }
    if (!req.body.idSkill.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }

    // Find the skill associated with the provided idSkill
    const skill = await Skill.findById(req.body.idSkill);

    // Check if the user is assigned to the project related to the skill
    let result = await userIsAssignedToProject(req.userId, skill.idProject.toString());
    if (result == -1)
        return res.status(500).send({ message: "Unable to find the account link and Project!" });
    if (result == 0)
        return res.status(403).send({ message: "Unable to change your note, you are not registered for this Project!" });

    // Find the UserSkill document using the userId and idSkill
    const filter = { idUser: req.userId, idSkill: req.body.idSkill };

    // Update the UserSkill document with the provided data
    UserSkill.findOneAndUpdate(filter, { $set: req.body }, { new: true, runValidators: true, useFindAndModify: false })
        .then(data => {
            if (!data) {
                res.status(400).send({ message: "Unable to update the skill. Maybe the skill was not found!" });
            } else {
                res.send({ message: "Your note has been changed!" });
            }
        })
        .catch(err => {
            res.status(500).send({ message: "Unable to update your note to skill!" });
        });
};

// Find user skills associated with a project
exports.findByProject = async (req, res) => {
    if (!req.body.idProject.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    // Find skills associated with the provided project id
    const listSkills = await Skill.find(req.body.idProject);
    const filter = { idUser: req.userId, idSkill: { $in: listSkills } };
    try {
        // Find UserSkill documents based on the filter
        const userSkills = await UserSkill.find(filter);

        // Convert UserSkill documents to JSON format
        const listeUserSkill = await Promise.all(
            userSkills.map(async (userSkill) => {
                return await userSkill.toJSON();
            })
        );

        res.send(listeUserSkill);
    } catch (err) {
        res.status(500).send({ message: err.message || "Unable to retrieve information from your skills!" });
    }
};

// Find user skills associated with a skill
exports.findBySkill = async (req, res) => {
    if (!req.params.idSkill.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    const filter = { idSkill: req.params.idSkill };
    try {
        // Find UserSkill documents based on the filter
        const userSkills = await UserSkill.find(filter);

        // Convert UserSkill documents to JSON format
        const listeUserSkill = await Promise.all(
            userSkills.map(async (userSkill) => {
                return await userSkill.toJSON();
            })
        );

        res.send(listeUserSkill);
    } catch (err) {
        res.status(500).send({ message: err.message || "Unable to retrieve information from your skills!" });
    }
};

// Check if the user is assigned to the project
async function userIsAssignedToProject(idUser, idProject) {
    try {
        var condition = { "idUser": idUser, "idProject": idProject };

        // Find the UserProject document based on the condition
        const data = await UserProject.findOne(condition);

        if (!data) {
            return 0; // User is not assigned to the project
        } else {
            if (data.status === "Affected") {
                return 1; // User is assigned to the project
            } else {
                return 0; // User is not assigned to the project
            }
        }
    } catch (err) {
        return -1; // Error occurred while checking user assignment to the project
    }
}
