const db = require("../models/index");

const User = db.user;
const Project = db.project;
const UserProject = db.userProject;
const UserSkill = db.userSkill;

var bcrypt = require("bcryptjs");

// Find all users
exports.findAll = (req, res) => {
    const title = req.query.searchByNom;
    var condition = title ? { nom: { $regex: new RegExp(title), $options: "i" } } : {};

    User.find(condition)
        .populate("roles", "-__v")
        .then(data => {
            const UsersFormatte = data.map((user) => {
                return user.toJSON();
            });

            res.send(UsersFormatte);
        })
        .catch(err => {
            res.status(500).send({
                message:
                    err.message || "An error occurred while retrieving Users"
            });
        });

};

// Find a single user by ID
exports.findOne = (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    User.findById(id)
        .populate("roles", "-__v")
        .then(data => {
            if (!data)
                res.status(404).send({ message: "User not found with ID: " + id });
            else
                res.send(data.toJSON());
        })
        .catch(err => {
            res
                .status(500)
                .send({ message: "Error extracting User with ID: " + id });
        });
};

// Update a user
exports.update = async (req, res) => {
    let Admin = false;

    if (Object.keys(req.body).length === 0) {
        return res.status(400).send({ message: "No data was provided" });
    }

    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }

    if (req.body.password) {
        req.body.password = bcrypt.hashSync(req.body.password, 10)
    }
    else {
        delete req.body.password
    }

    await User.findById(req.userId)
        .populate("roles", "-__v")
        .then(async data => {
            Admin = await data.needAdmin()
        })

    if (!Admin) {
        delete req.body.roles
        if (req.userId != id)
            return res.status(403).send({
                message: "You do not have the permissions to modify this User!"
            });
    }

    User.findByIdAndUpdate(id, { $set: req.body }, { new: true, runValidators: true, useFindAndModify: false })
        .populate("roles", "-__v")
        .then(data => {
            if (!data) {
                res.status(404).send({
                    message: `Unable to update user with id=${id}. Maybe user not found!`
                });
            } else res.send({ message: "User has been successfully updated", User: data.toJSON() });
        })
        .catch(err => {
            res.status(500).send({
                message: "User update error with id: " + id
            });
        });
};

// Delete a user
exports.delete = async (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    
    try {
        const user = await User.findById(id);
        if (!user) {
            return res.status(404).send({ message: `Unable to delete user with id=${id}. User may not have been found!` });
        }
        await UserSkill.deleteMany({ _id: user._id });
        await User.deleteOne({ _id: user._id });
        res.send({ message: "User has been deleted successfully!" });
    } catch (err) {
        res.status(500).send({ message: err.message || "Unable to delete user with id =" + id });
    }
};

// Find projects associated with a user
exports.findAffectToProject = (req, res) => {
    UserProject.find({ idUser: req.userId })
        .then(async data => {
            let listeUserProject = [];

            for (let userProject of data) {
                await Project.findById(userProject.idProject)
                    .then(async projects => {
                        listeUserProject.push(await projects.toJSON());
                    })
                    .catch(err => {
                        return res.status(500).send({ message: "Unable to retrieve requested information!" });
                    });
            }
            res.send(listeUserProject);
        })
        .catch(err => {
            return res.status(500).send({ message: "Unable to retrieve requested information!" });
        });
};
