const db = require("../models/index");
const Skill = db.skill;
const Project = db.project;
const UserSkill = db.userSkill;
const UserProject = db.userProject;

// Find all skills
exports.findAll = (req, res) => {
    Skill.find()
        .populate("idProject", "-__v")
        .then(data => {
            const skills = data.map((skill) => {
                return skill.toJSON();
            });

            res.send(skills);
        })
        .catch(err => {
            res.status(500).send({
                message: err.message || "An error occurred while retrieving skills."
            });
        });
};

// Find all skills by project
exports.findAllbyProject = (req, res) => {
    if (!req.params.id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    Skill.find({ idProject: req.params.id })
        .populate("idProject", "-__v")
        .then(data => {
            const skills = data.map((skill) => {
                return skill.toJSON();
            });

            res.send(skills);
        })
        .catch(err => {
            res.status(500).send({
                message: err.message || "An error occurred while retrieving skills."
            });
        });
};

// Find a single skill by ID
exports.findOne = (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    Skill.findById(id)
        .populate("idProject", "-__v")
        .then(data => {
            if (!data)
                res.status(404).send({ message: "Skill not found with ID: " + id });
            else
                res.send(data.toJSON());
        })
        .catch(err => {
            res.status(500).send({ message: "Error extracting competence with ID: " + id });
        });
};

// Update a skill
exports.update = async (req, res) => {
    if (Object.keys(req.body).length === 0) {
        return res.status(400).send({ message: "No data transmitted" });
    }

    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }

    delete req.body.idProject;

    Skill.findByIdAndUpdate(id, { $set: req.body }, { new: true, runValidators: true, useFindAndModify: false })
        .populate("idProject", "-__v")
        .then(data => {
            if (!data) {
                res.status(404).send({
                    message: `Unable to update competence with id=${id}. Maybe the skill hasn't been found!`
                });
            } else res.send({ message: "Skill successfully updated" });
        })
        .catch(err => {
            res.status(500).send({
                message: "Skill update error with id: " + id
            });
        });
};

// Delete a skill
exports.delete = async (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    try {
        const skill = await Skill.findById(id).populate("idProject", "-__v");
        if (!skill) {
            return res.status(404).send({ message: `Unable to delete skill with id=${id}. The skill may not have been found!` });
        }
        await UserSkill.deleteMany({ idSkill: skill._id });
        await Skill.deleteOne({ _id: skill._id });
        res.send({ message: "The skill was successfully removed!" });
    } catch (err) {
        res.status(500).send({ message: err.message || "Unable to delete skill with id =" + id });
    }
};

// Create a skill
exports.create = (req, res) => {
    const skill = new Skill({
        nom: req.body.nom,
        description: req.body.description,
        idProject: req.body.idProject
    });

    skill.save().then(async data => {
        const listUser = await UserProject.find({ idProject: req.body.idProject, status: "Affected" });
        for (let thisUser of listUser) {
            const userSkill = new UserSkill({
                idSkill: data._id,
                idUser: thisUser.idUser,
                note: "Non acquis"
            });
            userSkill.save().catch(err => {
                res.status(500).send({ message: "You can't save your skills!" });
            });
        }
        res.send({ message: "The skill has been successfully registered!" });
    })
        .catch(err => {
            res.status(500).send({ message: `Unable to register a skill at this time! ${err}` });
        });
};
