const db = require("../models/index");
const Project = db.project;
const UserProject = db.userProject;
const Skill = db.skill;
const UserSkill = db.userSkill;

// Find all projects
exports.findAll = async (req, res) => {
    try {
        const data = await Project.find();

        const projects = await Promise.all(
            data.map(async (project) => {
                return await project.toJSON();
            })
        );

        res.send(projects);
    } catch (err) {
        res.status(500).send({
            message:
                err.message || "An error occurred while retrieving the projects.",
        });
    }
};

// Find a single project by ID
exports.findOne = (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    Project.findById(id)
        .then(async data => {
            if (!data)
                res.status(404).send({ message: "Project not found with ID: " + id });
            else
                res.send(await data.toJSON());
        })
        .catch(err => {
            res
                .status(500)
                .send({ message: "Error in project extraction with ID: " + id });
        });
};

// Update a project
exports.update = async (req, res) => {
    if (Object.keys(req.body).length === 0) {
        return res.status(400).send({ message: "No data transmitted" });
    }
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    let result = await isOwner(req.userId, id);

    if (result == 1) {
        Project.findById(id)
            .then(async data => {
                if (!data) {
                    res.status(404).send({ message: `Unable to update project with id=${id}. Maybe the project hasn't been found!` });
                } else {
                    res.send(await data.toJSON())
                }
            })
            .catch(err => {
                res.status(500).send({ message: "Project not found with id: " + id });
            });
    } else {
        if (result == 0)
            res.status(401).send({ message: "The project cannot be modified. You don't own the project!" });
        else
            res.status(500).send({ message: "Unable to find the account and project link!" });
    }
};

// Delete a project
exports.delete = async (req, res) => {
    const id = req.params.id;
    if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }
    let result = await isOwner(req.userId, id);

    if (result == 1) {
        try {
            const project = await Project.findById(id);
            if (!project) {
                return res.status(404).send({ message: "Impossible to delete the project. The project has not been found!" });
            }
            const listSkills = await Skill.find({ idProject: project._id });
            for (let skill of listSkills) {
                await UserSkill.deleteMany({ idSkill: skill._id });
            }
            await Skill.deleteMany({ idProject: project._id });
            await Project.deleteOne({ _id: project._id });
            res.send({ message: "The project has been successfully deleted!" });
        } catch (err) {
            res.status(500).send({ message: err.message || "Can’t delete the project yet!" });
        }
    } else {
        if (result == 0)
            res.status(401).send({ message: "You can't delete the project. You don't own the project!" });
        else
            res.status(500).send({ message: "Unable to find the account and project link!" });
    }
};

// Create a project
exports.create = (req, res) => {
    // Check 'nom' field
    if (!req.body.nom) {
        return res.status(400).send({ message: "Name Project is required." });
    }

    const project = new Project({
        nom: req.body.nom,
        description: req.body.description
    });

    project.save().then(async data => {
        const userProject = new UserProject({
            idUser: req.userId,
            idProject: data._id,
            status: "Owner"
        });

        userProject.save().then(data => {
            res.send({ message: "The project was successfully registered!" });
        })
        .catch(err => {
            res.status(500).send({ message: `Unable to save project account link for now! ${err}` });
        });
    })
        .catch(err => {
            res.status(500).send({ message: `Impossible to register a project at the moment! ${err}` });
        });
};

// Register a user to a project
exports.registerToProject = async (req, res) => {
      const id = req.params.id;
  
      if (!id.match(/^[0-9a-fA-F]{24}$/)) {
        // No, it's not a valid ObjectId.
        return res.status(404).send({ message: "Bad format of Object ID!" });
    }

      let result = await isOwner(req.userId, id);
      if (result == 1) {
        return res.status(403).send({ message: "You are the owner of this project, so it is impossible to get involved!" });
      }
      if (result == -1) {
        return res.status(500).send({ message: "Unable to find the account and project link!" });
      }
  
      
        const dataProject = Project.findById(id);
        console.log(dataProject);
        
        if (!dataProject) {
            return res.status(404).send({ message: "Project not Found!" });
        }
        else{
            const userProject = new UserProject({
                idUser: req.userId,
                idProject: dataProject._id, // Assign the string value directly
                status: "Affected"
            });
        
            const savedUserProject = await userProject.save();
        
            const listLinkedSkills = await Skill.find({ idProject: id }); // Use the string value directly
            if (listLinkedSkills.length === 0) {
                return res.status(404).send({ message: "No Skill Found!" });
            }
        
            for (let skill of listLinkedSkills) {
                const userSkill = new UserSkill({
                idSkill: skill._id,
                idUser: req.userId,
                note: "Non acquis"
                });
        
                await userSkill.save();
            }
            return res.send({ message: "You are registered for the project: " + dataProject.nom + "!" });
            }
  };

// Check if the user is the owner of a project
async function isOwner(idUser, idProject) {
    try {
        var condition = { "idUser": idUser, "idProject": idProject };

        const data = await UserProject.findOne(condition);

        if (!data) {
            return 0;
        } else {
            if (data.status === "Owner") {
                return 1;
            } else {
                return 0;
            }
        }
    } catch (err) {
        console.log(err.message);
        return -1;
    }
}
